// SPDX-License-Identifier: GPL-2.0
// Enable EL0 access to PMU system registers on all CPUs.
// Tested on arm64 (Cortex-A76 class). Use at your own risk.

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/percpu.h>
#include <linux/cpu.h>
#include <linux/cpuhotplug.h>
#include <linux/smp.h>

MODULE_DESCRIPTION("Enable EL0 access to PMU sysregs (PMUSERENR_EL0.EN)");
MODULE_AUTHOR("you");
MODULE_LICENSE("GPL");

static int hp_state = CPUHP_INVALID;
static DEFINE_PER_CPU(u64, saved_pmuserenr);

/* Set EN bit (bit[0]) of PMUSERENR_EL0 on current CPU, saving previous value */
static void pmu_enable_el0_on_this_cpu(void *info)
{
	u64 val;
	/* Read current */
	asm volatile("mrs %0, pmuserenr_el0" : "=r"(val));
	this_cpu_write(saved_pmuserenr, val);

	/* Set EN (bit 0) to allow EL0 access */
	val |= 1ull;
	asm volatile("msr pmuserenr_el0, %0" :: "r"(val));
	asm volatile("isb");
}

/* Restore saved value on current CPU */
static void pmu_restore_on_this_cpu(void *info)
{
	u64 val = this_cpu_read(saved_pmuserenr);
	asm volatile("msr pmuserenr_el0, %0" :: "r"(val));
	asm volatile("isb");
}

/* Hotplug: when a CPU comes online, enable EL0 access on it as well */
static int pmu_hp_online(unsigned int cpu)
{
	u64 cur;
	asm volatile("mrs %0, pmuserenr_el0" : "=r"(cur));
	per_cpu(saved_pmuserenr, cpu) = cur;

	cur |= 1ull; /* EN */
	asm volatile("msr pmuserenr_el0, %0" :: "r"(cur));
	asm volatile("isb");
	pr_info("pmu_user_open: CPU%u enabled EL0 PMU access\n", cpu);
	return 0;
}

/* On CPU offline we do nothing special; kernel will restore on next online. */

static int __init pmu_user_open_init(void)
{
	int ret;

	/* Apply to all online CPUs now */
	on_each_cpu(pmu_enable_el0_on_this_cpu, NULL, 1);
	pr_info("pmu_user_open: EL0 PMU access enabled on online CPUs\n");

	/* Register hotplug callback */
	ret = cpuhp_setup_state_nocalls(CPUHP_AP_ONLINE_DYN, "pmu_user_open:online",
	                                pmu_hp_online, NULL);
	if (ret < 0) {
		pr_err("pmu_user_open: cpuhp_setup_state failed: %d\n", ret);
		/* best-effort rollback */
		on_each_cpu(pmu_restore_on_this_cpu, NULL, 1);
		return ret;
	}
	hp_state = ret;
	return 0;
}

static void __exit pmu_user_open_exit(void)
{
	if (hp_state != CPUHP_INVALID)
		cpuhp_remove_state_nocalls(hp_state);

	/* Restore original PMUSERENR_EL0 on all CPUs */
	on_each_cpu(pmu_restore_on_this_cpu, NULL, 1);
	pr_info("pmu_user_open: EL0 PMU access restored/disabled\n");
}

module_init(pmu_user_open_init);
module_exit(pmu_user_open_exit);
