`timescale 1ns/1ps
`default_nettype none

// SPI Master
module SPI_MASTER #(
  parameter SLAVE_DEVICE_NUM = 4,           // 接続するスレーブ・デバイスの数
  parameter MAX_TRANSFER_BITS = 16,         // 1 回の転送における SCLK サイクル数の最大値 (= 最大の転送ビット数)
  parameter CLKDIV_CNT_WIDTH = 4,           // クロック分周用カウンタのビット数
  parameter PRE_TRANSFER_SYSCLK_CYCLES = 4, // nCS をアサートしてから転送を開始 (SCLK 発振を開始) するまでの待ち時間 (sys_clk のサイクル数で指定 )
  parameter POST_TRANSFER_SYSCLK_CYCLES = 4 // 転送を終了 (SCLK 発振を停止) してから nCS をネゲートするまでの待ち時間 (sys_clk のサイクル数で指定)
) (
  input  wire sys_clk,   // クロック入力
  input  wire nrst,      // リセット入力 (負極性)

  // 設定信号 / 送信データ (sys_clkに同期)
  // "slave_device_sel" / "sclk_cycles" / "txd" は "start" がアサートされたサイクルで取り込まれる
  input wire start,                                       // 転送開始 (1サイクル・パルス信号)
  input wire [CLKDIV_CNT_WIDTH-1:0] sclk_div_ratio_half,  // spi_sclk 生成用の sys_clk の分周比率/2 (1以上の値を指定)
  input wire [$clog2(MAX_TRANSFER_BITS):0]  sclk_cycles,  // SCLKのサイクル数
  input wire cpol,                                        // SCLKの極性 (0:負極性, 1:正極性)
  input wire cpha,                                        // SCLKの位相
  input wire [$clog2(SLAVE_DEVICE_NUM)-1:0] slave_device_sel, // スレーブ・デバイスの選択
  input wire [MAX_TRANSFER_BITS-1:0] txd,                 // 送信データ

  // 受信データ (sys_clkに同期)
  // "rxd" は done 信号がアサートされたタイミングで有効となる
  output logic done,                        // 受信完了 (1サイクル・パルス信号)
  output logic [MAX_TRANSFER_BITS-1:0] rxd, // 受信データ

  // SPI インタフェース信号
  output logic spi_sclk,                      // クロック信号 (SCLK)
  output logic spi_mosi,                      // 出力ビット信号 (MOSI: Master-Out, Slave-In)
  input  wire  spi_miso,                      // 入力ビット信号 (MISO: Master-In, Slave-Out)
  output logic [SLAVE_DEVICE_NUM-1:0] spi_ncs // スレーブデバイス選択 (nCS : 負極性 Chip Select)
);

//== ステートマシン / 完了通知生成 ===============//
  // ステート番号の定義
  localparam ST_IDLE          = 2'd0,   // 動作停止状態
             ST_PRE_TRANSFER  = 2'd1,   // nCSをアサート後, SCLK出力を開始するまでの待機状態
             ST_TRANSFER      = 2'd2,   // 転送実行中状態 (SCLK出力中)
             ST_POST_TRANSFER = 2'd3;   // 転送完了後, nCSをネゲートするまでの待機状態
  
  logic [1:0] state;                                      // ステート記録レジスタ
  logic [15:0] sys_clk_cycle_count;                       // 入力クロック (sys_clk) のクロック・サイクル数をカウンタ
  logic [$clog2(MAX_TRANSFER_BITS)+1:0] sclk_edge_count;  // SCLKのエッジ数カウンタ
  logic done_reg;                                         // 完了通知信号

  always_ff@(posedge sys_clk) begin
    if(!nrst) begin
      state <= ST_IDLE;
      sys_clk_cycle_count <= 0;
      sclk_edge_count <= 0;
      done_reg <= 0;
    end else begin
      // sys_clkサイクル数カウンタをインクリメント
      sys_clk_cycle_count <= sys_clk_cycle_count + 16'b1;

      // 完了通知信号は動作が完了したサイクルでのみアサートするため、基本的には0
      done_reg <= 0;
      
      // Stateの遷移
      case (state) 
        ST_IDLE: begin 
          if(start) begin
            // start 信号がアサートされたら次状態へ遷移
            state <= ST_PRE_TRANSFER;  // 状態遷移
            sys_clk_cycle_count <= 0;  // sys_clkサイクル数カウンタをリセット
          end
        end
        ST_PRE_TRANSFER: begin
          // PRE_TRANSFER_SYSCLK_CYCLES で指定されたサイクル数待機したら次状態へ遷移
          if(sys_clk_cycle_count == PRE_TRANSFER_SYSCLK_CYCLES) begin
            state <= ST_TRANSFER;      // 状態遷移
            sys_clk_cycle_count <= 0;  // sys_clkサイクル数カウンタをリセット
            sclk_edge_count <= 0;      // SCLKのエッジ数カウンタをリセット
          end
        end
        ST_TRANSFER: begin
          // SCLK のエッジをカウント
          if(spi_sclk_toggle) sclk_edge_count <= sclk_edge_count + 1'b1;  

          // SCLKを sclk_cycles で指定したサイクル数出力したら次状態へ遷移
          if( sclk_edge_count == (sclk_cycles_reg << 1) ) begin  // エッジ数カウントなのでサイクル数を2倍して比較
            state <= ST_POST_TRANSFER; // 状態遷移
            sys_clk_cycle_count <= 0;  // sys_clkサイクル数カウンタをリセット
            done_reg <= 1;             // この時点で転送は完了するため完了通知をアサート (rxdのデータが有効)
          end 
        end
        ST_POST_TRANSFER: begin
          // POST_TRANSFER_SYSCLK_CYCLES で指定されたサイクル数待機したら動作停止状態に戻る
          if(sys_clk_cycle_count == PRE_TRANSFER_SYSCLK_CYCLES) begin
            state <= ST_IDLE;      // 状態遷移
          end
        end
        default: ;
      endcase
    end
  end

  assign done = done_reg;

  //== 入力設定信号の保持レジスタ ===============//
  logic [CLKDIV_CNT_WIDTH-1:0] sclk_div_ratio_half_reg;
  logic [$clog2(MAX_TRANSFER_BITS)+1:0]  sclk_cycles_reg;
  logic cpol_reg;
  logic cpha_reg;
  logic [$clog2(SLAVE_DEVICE_NUM)-1:0] slave_device_sel_reg;

  always_ff@(posedge sys_clk) begin
    if(!nrst) begin
      sclk_div_ratio_half_reg <= 0;
      sclk_cycles_reg <= 0;
      cpol_reg <= 0;
      cpha_reg <= 0;
      slave_device_sel_reg <= 0;
    end else begin
      if(state == ST_IDLE && start) begin
        // 動作停止状態のとき, start信号がアサートされたら取り込み
        sclk_div_ratio_half_reg <= sclk_div_ratio_half;
        sclk_cycles_reg <= sclk_cycles;
        cpol_reg <= cpol;
        cpha_reg <= cpha;
        slave_device_sel_reg <= slave_device_sel;
      end
    end
  end


  //== nCS 生成 ==========================//
  logic [SLAVE_DEVICE_NUM-1:0] spi_ncs_reg;

  always_ff@(posedge sys_clk) begin
    if(!nrst) begin
      spi_ncs_reg <= {SLAVE_DEVICE_NUM{1'b1}};  // All-1
    end else begin
      spi_ncs_reg <= {SLAVE_DEVICE_NUM{1'b1}};  // All-1

      if(state == ST_PRE_TRANSFER || state == ST_TRANSFER || state == ST_POST_TRANSFER) begin
        // 動作中状態のとき, slave_device_sel で選択されたデバイスのnCSを0にする
        spi_ncs_reg[slave_device_sel_reg] <= 1'b0;
      end
    end
  end

  assign spi_ncs = spi_ncs_reg;


//== SCLK 生成 (クロック分周回路) ======//
  logic [CLKDIV_CNT_WIDTH-1:0] clk_div_counter;   // 分周用カウンタ
  logic spi_sclk_reg;  // SCLK信号
  
  // SCLK生成用の条件判定
  wire spi_sclk_toggle  = ((clk_div_counter + 1) == sclk_div_ratio_half_reg); // カウンタが 分周比/2 になるときにSCLKを反転する
  wire spi_sclk_rising  = ~spi_sclk_reg & spi_sclk_toggle;                    // 次のsys_clkサイクルでSCLKの立ち上がりが発生
  wire spi_sclk_falling = spi_sclk_reg & spi_sclk_toggle;                     // 次のsys_clkサイクルでSCLKの立ち下がりが発生

  always_ff@(posedge sys_clk) begin
    if(!nrst) begin
      clk_div_counter <= 0;
      spi_sclk_reg <= 0;
    end else begin
      if(state == ST_TRANSFER) begin
        // 転送実行中のときSCLKを発振させる
        if(spi_sclk_toggle) begin
          // カウンタが 分周比/2 になるとき, SCLKを反転する
          spi_sclk_reg <= ~spi_sclk_reg;  // SCLKを反転
          clk_div_counter <= 0;           // カウンタ値を0に戻す
        end else begin
          // カウンタをインクリメント
          clk_div_counter <= clk_div_counter + 1'b1; 
        end
      end else begin
        // 転送実行中以外は0としておく
        clk_div_counter <= 0;
        spi_sclk_reg <= 0;
      end 
    end
  end

  // 出力SCLKの極性をCPOLによって切り替える (CPOL=0:アイドル時にLow, CPOL=1:アイドル時にHigh)
  assign spi_sclk = (cpol_reg==0)? spi_sclk_reg : ~spi_sclk_reg;


//== 出力(MOSI) シフト・レジスタ ======//
  logic [MAX_TRANSFER_BITS-1:0] mosi_shift_reg;
  always_ff@(posedge sys_clk) begin
    if(!nrst) begin
      mosi_shift_reg <= 0;
    end else begin
      if(state == ST_IDLE) begin
        // 動作開始時に送信データをシフト・レジスタにセット
        if(start) mosi_shift_reg <= txd;  
      end else
      if(state == ST_TRANSFER) begin
        // 転送中は、SCLKのエッジで 下位Bit -> 上位Bit 方向に1Bitシフトする 
        // (CPHA設定により立ち上がり/立ち下がりのどちらでシフトするかを選択)
        if( (cpha_reg==0 && spi_sclk_falling) || (cpha_reg==1 && spi_sclk_rising && sclk_edge_count != 0) ) begin
          mosi_shift_reg <= {mosi_shift_reg[MAX_TRANSFER_BITS-2:0], 1'b0};
        end
      end
    end
  end

  // シフト・レジスタの最上位BitをMOSIに出力する
  assign spi_mosi = mosi_shift_reg[MAX_TRANSFER_BITS-1];


//== 入力(MISO) シフト・レジスタ ======//
  logic [MAX_TRANSFER_BITS-1:0] miso_shift_reg;
  always_ff@(posedge sys_clk) begin
    if(!nrst) begin
      miso_shift_reg <= 0;
    end else begin
      if(state == ST_TRANSFER) begin
        // 転送中は、SCLKのエッジで 下位Bit -> 上位Bit 方向に1Bitシフトし、最下位BitにMISOの入力値を取り込む
        // (CPHA設定により立ち上がり/立ち下がりのどちらで取り込むかを選択)
        if( (cpha_reg==0 && spi_sclk_rising) || (cpha_reg==1 && spi_sclk_falling) ) begin
          miso_shift_reg <= {miso_shift_reg[MAX_TRANSFER_BITS-2:0], spi_miso};
        end
      end else begin
        miso_shift_reg <= 0;
      end
    end
  end

  assign rxd = miso_shift_reg;

endmodule

`default_nettype wire
