`timescale 1ns/1ps
`default_nettype none

module signal_processing #(
    parameter INPUT_BITS = 12,
    parameter OUTPUT_BITS = 12
) (
    input wire clk,      // クロック入力
    input wire nrst,     // リセット入力 (負極性)

    // 入力信号
    input wire din_valid,
    output wire din_ready, 
    input wire [INPUT_BITS-1:0] din_data,

    // 出力信号
    output logic dout_valid,
    input wire dout_ready,
    output logic [OUTPUT_BITS-1:0] dout_data
);
    //===== パラメータ設定 ===============================
    // ※ 各IPコアの設定に合致した値を記述する
    localparam FFT_POINT_N = 2048;                 // FFT(IFFT) Point数
    localparam FFT_INPUT_BITS = INPUT_BITS;        // FFT入力データのビット幅
    localparam FFT_OUTPUT_BITS = 12;               // FFT出力データのビット幅
    localparam FIFO_DEPTH = FFT_POINT_N * 2;       // FIFOのDepth (FIFO IPの設定と合わせる)

    //===== モジュール入出力信号の宣言 ====================
    // 入力データ用FIFO 入出力信号
    logic [$clog2(FIFO_DEPTH):0] fifo_input_readable_n; 
    logic fifo_input_full;
    logic [INPUT_BITS-1:0] fifo_input_dout;

    // FFT 入出力信号
    logic fft_start, fft_busy;
    logic fft_sod, fft_eod, fft_ipd;
    logic [$clog2(FFT_POINT_N)-1:0] fft_idx;
    logic [FFT_INPUT_BITS-1:0] fft_din_re, fft_din_im;
    logic fft_soud, fft_eoud, fft_opd;
    logic [FFT_OUTPUT_BITS-1:0] fft_dout_re, fft_dout_im;

    // 周波数フィルタ 入出力信号
    logic filter_dout_valid;
    logic [FFT_OUTPUT_BITS-1:0] filter_dout_re, filter_dout_im;

    // FFT+周波数フィルタ結果格納用FIFO 入出力信号
    logic fifo_fftres_full;
    logic [$clog2(FIFO_DEPTH):0] fifo_fftres_occupancy, fifo_fftres_readable_n;
    logic [FFT_OUTPUT_BITS-1:0] fifo_fftres_dout_re, fifo_fftres_dout_im;
    logic [$clog2(FIFO_DEPTH):0] fifo_fftres_writable_n;

    // IFFT 入出力信号
    logic ifft_start, ifft_busy;
    logic ifft_sod, ifft_eod, ifft_ipd;
    logic [$clog2(FFT_POINT_N)-1:0] ifft_idx;
    logic [FFT_OUTPUT_BITS-1:0] ifft_din_re, ifft_din_im;
    logic ifft_soud, ifft_eoud, ifft_opd;
    logic [FFT_OUTPUT_BITS-1:0] ifft_dout_re, ifft_dout_im;

    // 出力データ用FIFO 入出力信号
    logic fifo_output_empty;
    logic fifo_output_rden;
    logic [$clog2(FIFO_DEPTH):0] fifo_output_occupancy;
    logic [$clog2(FIFO_DEPTH):0] fifo_output_writable_n;
    logic [OUTPUT_BITS-1:0] fifo_output_dout;


    //===== 各モジュールのインスタンスと接続 =================
    // 入力IF信号
    assign din_ready = ~fifo_input_full; // 入力データ用FIFOがFullでなければ, データを受信可能

    // 入力データ用FIFO (実数値のみ)
    fifo_real fifo_input (
        .Reset(~nrst), 
        .RdClk(clk), 
        .WrClk(clk),
        .WrEn(din_valid),
        .Data(din_data),
        .RdEn(fft_ipd),
        .Q(fifo_input_dout),
        .Empty(),
        .Full(fifo_input_full),
        .Wnum(),
        .Rnum(fifo_input_readable_n)
    );

    // FFT機能の起動制御用のステートマシン
    localparam FFT_ST_IDLE     = 2'h0;
    localparam FFT_ST_WAITBUSY = 2'h1;
    localparam FFT_ST_BUSY     = 2'h2;

    logic [1:0] fft_state;
    always_ff@(posedge clk) begin
        if(~nrst) begin
            fft_state <= FFT_ST_IDLE;
        end else begin
            case(fft_state)
            FFT_ST_IDLE: begin
                if (fft_start == 'b1) fft_state <= FFT_ST_WAITBUSY;
            end
            FFT_ST_WAITBUSY: begin
                if (fft_busy == 'b1) fft_state <= FFT_ST_BUSY;
            end
            FFT_ST_BUSY: begin
                if (fft_busy == 'b0) fft_state <= FFT_ST_IDLE;
            end
            default: begin
                fft_state <= FFT_ST_IDLE;
            end 
            endcase
        end
    end

    // FFT機能の入力信号
    always_comb begin
        // FFT IPの起動信号
        //   - 条件1: FFT IPが未動作状態
        //   - 条件2: 入力側のFIFOにFFT Point数以上のデータがある
        //   - 条件3: 出力側のFIFOにFFT Point数以上の空きがある
         fft_start = 
             (fft_state == FFT_ST_IDLE) &&             
             (fifo_input_readable_n >= FFT_POINT_N) && 
             (fifo_fftres_writable_n >= FFT_POINT_N); 

        // FFTの実部入力値はFIFOから取り出す
        fft_din_re = fifo_input_dout;

        // FFTの虚部入力値は0 (今回は実数データ入力を対象とするため)
        fft_din_im = 'b0;
    end

    // FFT機能 本体
    FFT_Top  fft_top_inst (
        .clk(clk),
        .rst(~nrst),
        .start(fft_start),
        .busy(fft_busy),
        .sod(fft_sod),
        .eod(fft_eod),
        .ipd(fft_ipd),
        .idx(fft_idx),
        .xn_re(fft_din_re),
        .xn_im(fft_din_im),
        .soud(fft_soud),
        .eoud(fft_eoud),
        .opd(fft_opd),
        .xk_re(fft_dout_re),
        .xk_im(fft_dout_im)
    );

    // 周波数フィルタモジュール
    freq_filter #( 
        .DATA_BITS(INPUT_BITS), 
        .COEF_TOTAL_BITS(8), 
        .COEF_FRAC_BITS(4), 
        .FFT_N(FFT_POINT_N) 
    ) freq_filter_inst (
        .clk(clk), 
        .nrst(nrst),
        .din_valid(fft_opd),
        .din_re_data(fft_dout_re),
        .din_im_data(fft_dout_im),
        .dout_valid(filter_dout_valid),
        .dout_re_data(filter_dout_re),
        .dout_im_data(filter_dout_im)
    );


    // 中間(FFT->周波数フィルタ) 結果格納用FIFO (複素数: Data[11:0]:実部, Data[23:12]:虚部)
    always_comb begin
        fifo_fftres_writable_n = FIFO_DEPTH - fifo_fftres_occupancy;
    end
    
    fifo_complex fifo_fftres (
        .Reset(~nrst), 
        .RdClk(clk), 
        .WrClk(clk),
        .WrEn(filter_dout_valid),
        .Data({filter_dout_im, filter_dout_re}),
        .RdEn(ifft_ipd),
        .Q({fifo_fftres_dout_im, fifo_fftres_dout_re}),
        .Empty(),
        .Full(fifo_fftres_full),
        .Wnum(fifo_fftres_occupancy),
        .Rnum(fifo_fftres_readable_n)
    );

    // IFFT機能の起動制御用のステートマシン
    localparam IFFT_ST_IDLE     = 2'h0;
    localparam IFFT_ST_WAITBUSY = 2'h1;
    localparam IFFT_ST_BUSY     = 2'h2;

    logic [1:0] ifft_state;
    always_ff@(posedge clk) begin
        if(~nrst) begin
            ifft_state <= IFFT_ST_IDLE;
        end else begin
            case (ifft_state)
            IFFT_ST_IDLE: begin
                if (ifft_start == 'b1) ifft_state <= IFFT_ST_WAITBUSY;
            end
            IFFT_ST_WAITBUSY: begin
                if (ifft_busy == 'b1) ifft_state <= IFFT_ST_BUSY;
            end
            IFFT_ST_BUSY: begin
                if (ifft_busy == 'b0) ifft_state <= IFFT_ST_IDLE;
            end
            default: begin
                ifft_state <= IFFT_ST_IDLE;
            end 
            endcase
        end
    end

    // IFFT機能の入力信号
    always_comb begin
        // IFFT IPの起動
        //  - 条件1: IFFT IPが未動作状態
        //  - 条件2: 入力側のFIFOにIFFT Point数以上のデータがある
        //  - 条件3: 出力側のFIFOにIFFT Point数以上の空きがある
         ifft_start = 
             (ifft_state == IFFT_ST_IDLE) &&             
             (fifo_fftres_readable_n >= FFT_POINT_N) &&  
             (fifo_output_writable_n >= FFT_POINT_N) ;

        // IFFTの実部/虚部入力値をFIFOから取り出す
        ifft_din_re = fifo_fftres_dout_re;
        ifft_din_im = fifo_fftres_dout_im;
    end

    // IFFT機能 本体
    IFFT_Top  ifft_top_inst (
        .clk(clk),
        .rst(~nrst),
        .start(ifft_start),
        .busy(ifft_busy),
        .sod(ifft_sod),
        .eod(ifft_eod),
        .ipd(ifft_ipd),
        .idx(ifft_idx),
        .xn_re(ifft_din_re),
        .xn_im(ifft_din_im),
        .soud(ifft_soud),
        .eoud(ifft_eoud),
        .opd(ifft_opd),
        .xk_re(ifft_dout_re),
        .xk_im(ifft_dout_im)
    );


    // 出力データ用FIFO (実数値のみ)
    always_comb begin
        fifo_output_rden = dout_valid & dout_ready;
        fifo_output_writable_n = FIFO_DEPTH - fifo_fftres_occupancy;
    end
    
    fifo_real fifo_output (
        .Reset(~nrst), 
        .RdClk(clk), 
        .WrClk(clk),
        .WrEn(ifft_opd),
        .Data(ifft_dout_re),
        .RdEn(fifo_output_rden),
        .Q(fifo_output_dout),
        .Empty(fifo_output_empty),
        .Full(),
        .Wnum(fifo_output_occupancy),
        .Rnum()
    );

    // 出力IF信号
    assign dout_valid = ~fifo_output_empty;
    assign dout_data = fifo_output_dout;

endmodule

`default_nettype wire