`timescale 1ns/1ps
`default_nettype none

module spi_adc_sampling(
    input wire clk,      // クロック入力
    input wire nrst,     // リセット入力 (負極性)

    // 設定信号
    input wire [15:0] sampling_inverval, // サンプリング周期 (単位:クロックサイクル)

    // ADC用 SPI IF (入力(MISO)のみ)
    output logic adc_spi_sclk, 
    output logic adc_spi_ncs,  
    input  wire  adc_spi_miso,  

    // サンプリングデータ出力 (AXI-Stream方式)
    output logic dout_valid,
    input wire dout_ready,
    output logic [11:0] dout_data  // dout_valid=1 の時に有効
);

    // ADCサンプリング起動信号を生成するためのタイマ
    logic [15:0] sampling_timer;
    
    always_ff@(posedge clk) begin
        if (~nrst) begin
            sampling_timer <= 'b0;
        end else begin
            // 1クロックサイクルごとにタイマ値をデクリメント
            sampling_timer <= sampling_timer - 'b1;
            
            // タイマ値が0の時, 初期値に戻す (サンプリング周期設定信号から取得)
            if (sampling_timer == 0) begin
                sampling_timer <= (sampling_inverval - 'b1);
            end
        end
    end

    // タイマが0になり、かつ前のサンプリングデータの送信が完了していれば、
    // ADC受信用の SPI マスタ機能を起動する
    wire adc_start = (sampling_timer == 'b0) & (dout_valid_reg == 'b0);

    // ADC受信用のSPI マスタ機能を起動する
    localparam [3:0] adc_clk_div = 4;      // SCLKの周波数はクロック周波数の1/4 (例: クロック周波数 = 50 MHzのとき -> SCLK = 12.5 MHz)
    localparam [4:0] adc_sclk_cycle = 16;  // 1回の転送は16Bit
    localparam [0:0] adc_cpol = 1;         // アイドル時のSCLKはHighレベル
    localparam [0:0] adc_cpha = 0;         // SCLKの立ち下がりでキャプチャ, 立ち上がりでシフト

    logic adc_done;
    logic [15:0] adc_rxd;

    SPI_MASTER #(
        .SLAVE_DEVICE_NUM(1), .MAX_TRANSFER_BITS(16), .CLKDIV_CNT_WIDTH(4), 
        .PRE_TRANSFER_SYSCLK_CYCLES(1), .POST_TRANSFER_SYSCLK_CYCLES(1)
    ) spi_master_adc (
        .sys_clk(clk), .nrst(nrst),

        .start(adc_start),  // タイマで生成した起動信号を入力
        .sclk_div_ratio_half({adc_clk_div/2}[3:0]),
        .sclk_cycles(adc_sclk_cycle[4:0]),
        .cpol(adc_cpol),
        .cpha(adc_cpha),
        .slave_device_sel('b0),
        .txd('b0),         // ADCは入力のみなので出力データ無し(ダミーで0を入力)

        .done(adc_done),
        .rxd(adc_rxd),

        .spi_sclk(adc_spi_sclk),
        .spi_mosi(), // ADCは入力のみなので、MOSIは未接続
        .spi_miso(adc_spi_miso),
        .spi_ncs(adc_spi_ncs)
    );


    // 出力データレジスタ & 出力部
    logic dout_valid_reg;   // 出力データ格納レジスタ有効
    logic [11:0] dout_data_reg;  // 出力データ格納レジスタ

    always_ff@(posedge clk) begin
        if(~nrst) begin
            dout_valid_reg <= 'b0;
            dout_data_reg <= 'b0;
        end else begin
            if ((adc_done == 'b1) && (dout_valid_reg == 'b0)) begin
                // SPI受信が完了し、かつレジスタが空いていれば、SPI受信データを格納
                dout_valid_reg <= 'b1;
                dout_data_reg <= adc_rxd;
            end
            else if (dout_ready == 'b1) begin
                // 送信先がデータ受信したら、レジスタを空ける
                dout_valid_reg <= 'b0;
            end
        end
    end 

    assign dout_valid = dout_valid_reg;
    assign dout_data = dout_data_reg;

endmodule

`default_nettype wire
