`timescale 1ns/1ps
`default_nettype none

module spi_dac_output(
    input wire clk,      // クロック入力
    input wire nrst,     // リセット入力 (負極性)

    // 設定信号
    input wire [15:0] output_inverval, // 出力周期 (単位:クロックサイクル)

    // DAC用 SPI IF (出力(MOSI)のみ)
    output logic dac_spi_sclk, 
    output logic dac_spi_ncs,  
    output logic dac_spi_mosi,  

    // サンプリングデータ入力
    input wire din_valid,
    output logic din_ready,
    input wire [11:0] din   // din_valid=1 の時に有効
);

    // 出力周期信号を生成するためのタイマ
    logic [15:0] output_timer;
    
    always_ff@(posedge clk) begin
        if (~nrst) begin
            output_timer <= 'b0;
        end else begin
            // 1クロックサイクルごとにタイマ値をデクリメント
            output_timer <= output_timer - 'b1;
            
            // タイマ値が0の時, 初期値に戻す (出力周期設定信号から取得)
            if (output_timer == 0) begin
                output_timer <= (output_inverval - 'b1);
            end
        end
    end
 
    // タイマが0になり、かつ入力データが有効である時に、DAC出力を行う
    wire dac_start = (output_timer == 'b0) & din_valid;

    // DAC出力の開始と同時に、入力データを取得
    assign din_ready = dac_start;

    // DAC用のSPI Masterインタフェース
    localparam [3:0] dac_clk_div = 4;      // SCLKの周波数はクロック周波数の1/4 (例: クロック周波数 = 50 MHzのとき -> SCLK = 12.5 MHz)
    localparam [4:0] dac_sclk_cycle = 16;  // 1回の転送は16Bit
    localparam [0:0] dac_cpol = 0;         // アイドル時のSCLKはHighレベル
    localparam [0:0] dac_cpha = 1;         // SCLKの立ち下がりでキャプチャ, 立ち上がりでシフト

    logic dac_done;
    wire [1:0] dac_pd = 2'b00; // Power-Down Modeは使わない
    wire [15:0] dac_txd = {2'b00, dac_pd, din};  // [15:14]=RSV, [13:12]=PD, [11:0]=Data

    SPI_MASTER #(
        .SLAVE_DEVICE_NUM(1), .MAX_TRANSFER_BITS(16), .CLKDIV_CNT_WIDTH(4), 
        .PRE_TRANSFER_SYSCLK_CYCLES(1), .POST_TRANSFER_SYSCLK_CYCLES(1)
    ) spi_master_dac (
        .sys_clk(clk), .nrst(nrst),

        .start(dac_start),
        .sclk_div_ratio_half({dac_clk_div/2}[3:0]),
        .sclk_cycles(dac_sclk_cycle[4:0]),
        .cpol(dac_cpol),
        .cpha(dac_cpha),
        .slave_device_sel('b0),
        .txd(dac_txd),  

        .done(dac_done),
        .rxd(),      // DACは出力のみなので入力データ無し

        .spi_sclk(dac_spi_sclk),
        .spi_mosi(dac_spi_mosi),
        .spi_miso('b0), // DACは出力のみなので入力データ無し(ダミーで0を入力)
        .spi_ncs(dac_spi_ncs)
    );

endmodule

`default_nettype wire
